//
//  GHDeviceManagerProtocols.h
//  Garmin Health
//
//  Created by Echeandia, Augie on 8/8/18.
//  Copyright © 2018 Garmin International. All rights reserved.
//

#import "GHIntervalDuration.h"

@class GHDevice;
@class GHPairingCompletion;
@class GHScannedDevice;
@class GHBiometricsData;


#pragma mark -
/**
 *  Protocol for listening to device connection/disconnection.
 */
@protocol GHDeviceConnectionDelegate <NSObject>

/**
 *  Device connection is ready.
 *  @param device   The device that connected.
 */
- (void)didConnectDevice:(GHDevice *)device;
/**
 *  Device connection has ended.
 *  @param  device    Disconnected device.
 */
- (void)didDisconnectDevice:(GHDevice *)device;


@end

#pragma mark -
/**
 *  Used to notify of completion, failure, or intermediate states during pairing.
 */
@protocol GHPairingDelegate <NSObject>

/**
 *  Returns newly paired device
 *  @param device  The GHDevice that was paired with.
 */
- (void)didPairDevice:(GHDevice *)device;

/**
 *  Failed to pair device
 *  @param deviceId The device ID for the device that failed to pair.
 *  @param error  The error that occured.
 */
- (void)didFailToPairDevice:(NSUUID *)deviceId error:(NSError *)error;

/**
 *  Pairing is paused to allow for settings updates or device reset.
 *  @param completion  The GHPairingCompletion block that is called with the results of the call.
 */
- (void)didPausePairing:(GHPairingCompletion *)completion;


@end

#pragma mark -
/**
 *  Protocol for setting duration of results.
 */
@protocol GHIntervalDurationDelegate <NSObject>

@optional
/**
 *  Return the GHIntervalDuration to be used for the GHWellnessData results.
 *  Default of GHIntervalDurationOneMinute will be used if not provided.
 *  @param deviceId  The id of the device the wellness interval is being requested for.
 */
- (GHIntervalDuration)wellnessIntervalForDeviceID:(NSUUID *)deviceId;


@end

#pragma mark -
/**
 *  Protocol for listening for syncs.
 */
@protocol GHSyncDelegate <NSObject>

@optional
/**
 *  Sync started for device
 *  @param deviceId Identifier of syncing device.
 */
- (void)didStartSyncWithDevice:(NSUUID *)deviceId;

/**
 *  Current sync progress for device
 *  @param progress Estimated progress of sync.
 *  @param deviceId Identifier of syncing device.
 */
- (void)didReceiveSyncProgress:(double)progress deviceId:(NSUUID *)deviceId;

/**
 *  Sync completed for device
 *  @param deviceId Identifier of syncing device.
 *  @param error    Any errors that occurred during syncing.
 */
- (void)didCompleteSyncWithDevice:(NSUUID *)deviceId error:(NSError *)error;

/**
 *  Sync stalled with the device because it is communicating with Garmin Connect Mobile
 *  @param deviceId Identifier of syncing device.
 *  @param shouldStop block. Set shouldStop to YES to retry NO to end.
 */
- (void)didStallSyncWithDevice:(NSUUID *)deviceId completion:(void(^)(BOOL stop))shouldStop;

/**
 *  Received biometric data from the device
 *  @param deviceId Identifier of syncing device.
 *  @param biometricsData Data received from device.
 */
- (void)didCompleteSyncWithDevice:(NSUUID *)deviceId biometricsData:(GHBiometricsData *)biometricsData;


@end

#pragma mark -
/**
 *  Protocol for discovering GHScannedDevices.
 */
@protocol GHScanDelegate <NSObject>

@required
/**
 *  Scanner found a device.
 *  @param device   The device that the scanner located.
 */
- (void)didScanDevice:(GHScannedDevice *)device;

@optional
/**
 *  An error occured while scanning.
 *  @param error   The error that occured. Will Stop the scanner
 */
- (void)scanDidFailWithError:(NSError *)error;


@end
